<?php
session_start();
require_once 'config/database.php';
require_once 'includes/security.php';

class AdminLogin {
    private $db;
    private $security;
    
    public function __construct() {
        $this->db = new Database();
        $this->security = new Security();
    }
    
    public function login($username, $password) {
        // Input validation
        if (empty($username) || empty($password)) {
            return ['success' => false, 'message' => 'Please fill all fields'];
        }
        
        // Rate limiting
        if ($this->security->checkRateLimit($username)) {
            return ['success' => false, 'message' => 'Too many attempts. Try again later.'];
        }
        
        try {
            $sql = "SELECT id, username, password_hash, full_name, email, role, 
                           is_active, failed_attempts, last_login 
                    FROM admin_users 
                    WHERE username = ? AND is_active = 1";
            $params = [$username];
            $stmt = $this->db->prepare($sql, $params);
            
            if ($this->db->execute($stmt) && $user = $this->db->fetchArray($stmt)) {
                if (password_verify($password, $user['password_hash'])) {
                    // Check if password needs rehashing
                    if (password_needs_rehash($user['password_hash'], PASSWORD_DEFAULT)) {
                        $newHash = password_hash($password, PASSWORD_DEFAULT);
                        $this->updatePasswordHash($user['id'], $newHash);
                    }
                    
                    // Reset failed attempts
                    $this->resetFailedAttempts($user['id']);
                    
                    // Update last login
                    $this->updateLastLogin($user['id']);
                    
                    // Create session
                    $_SESSION['admin_logged_in'] = true;
                    $_SESSION['admin_id'] = $user['id'];
                    $_SESSION['admin_username'] = $user['username'];
                    $_SESSION['admin_role'] = $user['role'];
                    $_SESSION['admin_name'] = $user['full_name'];
                    $_SESSION['login_time'] = time();
                    
                    // Regenerate session ID for security
                    session_regenerate_id(true);
                    
                    // Log the login
                    $this->security->logActivity($user['id'], 'LOGIN_SUCCESS', 'User logged in successfully');
                    
                    return ['success' => true, 'message' => 'Login successful'];
                    
                } else {
                    // Increment failed attempts
                    $this->incrementFailedAttempts($user['id']);
                    $this->security->logActivity($user['id'], 'LOGIN_FAILED', 'Failed login attempt');
                    
                    return ['success' => false, 'message' => 'Invalid credentials'];
                }
            } else {
                $this->security->logActivity(0, 'LOGIN_FAILED', "Failed login attempt for username: $username");
                return ['success' => false, 'message' => 'Invalid credentials'];
            }
            
        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            return ['success' => false, 'message' => 'System error. Please try again.'];
        }
    }
    
    private function updatePasswordHash($userId, $hash) {
        $sql = "UPDATE admin_users SET password_hash = ? WHERE id = ?";
        $params = [$hash, $userId];
        $stmt = $this->db->prepare($sql, $params);
        if ($stmt) {
            $this->db->execute($stmt);
        }
    }
    
    private function resetFailedAttempts($userId) {
        $sql = "UPDATE admin_users SET failed_attempts = 0 WHERE id = ?";
        $params = [$userId];
        $stmt = $this->db->prepare($sql, $params);
        if ($stmt) {
            $this->db->execute($stmt);
        }
    }
    
    private function incrementFailedAttempts($userId) {
        $sql = "UPDATE admin_users SET failed_attempts = failed_attempts + 1 WHERE id = ?";
        $params = [$userId];
        $stmt = $this->db->prepare($sql, $params);
        if ($stmt) {
            $this->db->execute($stmt);
        }
    }
    
    private function updateLastLogin($userId) {
        $sql = "UPDATE admin_users SET last_login = GETDATE() WHERE id = ?";
        $params = [$userId];
        $stmt = $this->db->prepare($sql, $params);
        if ($stmt) {
            $this->db->execute($stmt);
        }
    }
}

// Generate CSRF token
$security = new Security();
$csrf_token = $security->generateCSRFToken();

// Handle login request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $security->verifyCSRFToken();
    
    $login = new AdminLogin();
    
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    
    $result = $login->login($username, $password);
    
    if ($result['success']) {
        header('Location: admin_dashboard.php');
        exit;
    } else {
        $error_message = $result['message'];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login</title>
    <link href="css/style.css" rel="stylesheet">

    <style>
        .login-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .login-card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
    <div class="login-container d-flex align-items-center">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-md-6 col-lg-4">
                    <div class="login-card p-4">
                        <div class="text-center mb-4">
                            <h2 class="fw-bold">Admin Login</h2>
                            <p class="text-muted">Access your administration panel</p>
                        </div>
                        
                        <?php if (isset($error_message)): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <?php echo htmlspecialchars($error_message); ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            
                            <div class="mb-3">
                                <label for="username" class="form-label">Username</label>
                                <input type="text" class="form-control" id="username" name="username" 
                                       required autofocus>
                            </div>
                            
                            <div class="mb-3">
                                <label for="password" class="form-label">Password</label>
                                <input type="password" class="form-control" id="password" name="password" 
                                       required>
                            </div>
                            
                            <div class="d-grid">
                                <button type="submit" class="btn btn-primary btn-lg">Sign In</button>
                            </div>
                        </form>
                        
                        <div class="text-center mt-3">
                            <small class="text-muted">
                                <i class="fas fa-shield-alt me-1"></i>
                                Secure Admin Access
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="js/jsscript2.js"></script>
    <script src="js/jsscript1.js"></script>
    
</body>
</html>